<?php
// This script is intended to be included within admin_dashboard.php,
// ensuring $conn (database connection) and $_SESSION['user_id'] are available.

// Default to user ID 1 for safety if session variable isn't set
$user_id = $_SESSION['user_id'] ?? 1;

$submitted_reports = [];
$status_message = '';

// --- 1. Fetch Submitted Reports ---
// We join with the users table twice: once for the submitted_by_user (which we know is $user_id)
// and once for the supervisor_user_id (to show the supervisor's name).
$sql_reports = "
    SELECT 
        ma.report_id, 
        ma.report_month, 
        ma.title, 
        ma.status, 
        ma.submission_date,
        s.username AS supervisor_name
    FROM 
        monthly_activities ma
    JOIN 
        users s ON ma.supervisor_user_id = s.user_id
    WHERE 
        ma.submitted_by_user_id = ? 
        AND ma.status != 'Draft'
    ORDER BY 
        ma.submission_date DESC
";

if ($stmt = $conn->prepare($sql_reports)) {
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $submitted_reports[] = $row;
        }
    } else {
        $status_message = "<div class='alert alert-info'>You have not submitted any reports yet.</div>";
    }
    
    $stmt->close();
} else {
    $status_message = "<div class='alert alert-error'>Database error preparing statement: " . $conn->error . "</div>";
}

// Helper function to render status badges
function get_status_badge($status) {
    $class = 'status-badge';
    switch ($status) {
        case 'Pending':
            $class .= ' status-pending';
            break;
        case 'Approved':
            $class .= ' status-approved';
            break;
        case 'Rejected':
            $class .= ' status-rejected';
            break;
        default:
            $class .= ' status-default';
    }
    return "<span class='{$class}'>" . htmlspecialchars($status) . "</span>";
}

?>

<style>
    /* Styling for the table and badges */
    .data-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
        box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        border-radius: 8px;
        overflow: hidden; /* Ensures rounded corners apply to content */
    }
    .data-table th, .data-table td {
        padding: 12px 15px;
        text-align: left;
        border-bottom: 1px solid #ddd;
    }
    .data-table th {
        background-color: #3498db;
        color: white;
        font-weight: 600;
        text-transform: uppercase;
    }
    .data-table tbody tr:nth-child(even) {
        background-color: #f9f9f9;
    }
    .data-table tbody tr:hover {
        background-color: #f1f1f1;
    }

    /* Status Badges */
    .status-badge {
        display: inline-block;
        padding: 5px 10px;
        border-radius: 12px;
        font-weight: bold;
        font-size: 0.85em;
        text-transform: capitalize;
        min-width: 80px;
        text-align: center;
    }

    .status-pending {
        background-color: #f1c40f; /* Yellow */
        color: #333;
    }

    .status-approved {
        background-color: #2ecc71; /* Green */
        color: white;
    }

    .status-rejected {
        background-color: #e74c3c; /* Red */
        color: white;
    }

    .status-default {
        background-color: #bdc3c7; /* Gray */
        color: #333;
    }

    .form-card {
        padding: 20px;
        border-radius: 8px;
        background: #fff;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
    }
</style>

<div class="form-card">
    <h2 class="section-header" style="color: #34495e; border-bottom: 2px solid #3498db; padding-bottom: 10px; margin-bottom: 20px;">
        <i class="fas fa-list-alt"></i> My Submitted Activity Reports
    </h2>
    
    <?php echo $status_message; ?>

    <?php if (!empty($submitted_reports)): ?>
        <table class="data-table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Date</th>
                    <th>Report Title</th>
                    <th>Supervisor</th>
                    <th>Status</th>
                    <th>Submitted On</th>
                    <th>Action</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($submitted_reports as $report): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($report['report_id']); ?></td>
                        <td><?php echo htmlspecialchars($report['report_month']); ?></td>
                        <td><?php echo htmlspecialchars($report['title']); ?></td>
                        <td><?php echo htmlspecialchars($report['supervisor_name']); ?></td>
                        <td><?php echo get_status_badge($report['status']); ?></td>
                        <td><?php echo htmlspecialchars(substr($report['submission_date'], 0, 10)); ?></td>
                        <td>
                            <!-- Action button - View details (assuming a view page will be created later) -->
                            <button class="btn-view" 
                                style="background-color: #3498db; color: white; padding: 5px 10px; border-radius: 4px; border: none;">
                                View
                            </button>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>
